--The scripts should always contain at least 10 functions :ScriptID, DisplaySourceName(), SourceSuperClassID(), SourceClassID(), DisplayDestinationName() DestinationSuperClassID(), DestinationClassID(), AboutText() and DefaultConversion, Conversion taking a param.
--Script ID that will append to destination
fn ScriptID = 
(
	""
)

--Return the name to be display for the source in the Scene Converter UI
fn DisplaySourceName =
(
	"mr Sun and Sky" 
)

--Returns the source class of the super class id 
fn SourceSuperClassID =
(
	80 --Helper 
)
--Returns the class id of the source class we use as a source for conversion
fn SourceClassID =
(
	#(0x4a1e6debL, 0x31c77d57L) --DaylightAssemblyHead
)

--Return the name to be display for the destination in the Scene Converter UI - must be in the order than the destination class ids
fn DisplayDestinationName =
(
	"Sun Positioner"
)

--Returns the destination class of the super class id
fn DestinationSuperClassID =
(
	48 --Light
)
--Returns the class id of the class we use as a destination for conversion
-- so we convert from SourceClassID() to DestinationClassID()
fn DestinationClassID =
(
	#(0x27fe329cL, 0x716f5b25L) --Sun_Positioner
)

--Validate that the source provided match with one of the SourceClass
fn VerifySource source =
(
	local ret = false	
	local src = SourceClassID()
	
	if(source == undefined) do return false
	
	if(classof src[1] == Array) then
    (
		for cd in src do
		(
			if((cd as string) == (source.ClassID as string)) do
			(
				ret = true
			)
		)
	)
    else
	(
		if((src as string) == (source.ClassID as string)) do
		(
			ret = true
		)
	)
    ret 
)
--Validate that the destination provided match with one of the DestinationClass
fn VerifyDestination destination =
(
	local ret = false	
	local dest = DestinationClassID()

	if(destination == undefined) do return false
	
	if(classof dest[1] == Array) then
    (
		for cd in dest do
		(
			if((cd as string) == (destination.ClassID as string)) do
			(
				ret = true
			)
		)
	)
    else
	(
		if((dest as string) == (destination.ClassID as string)) do
		(
			ret = true
		)
	)
    ret 
)
--Returns some information about this conversion script
fn AboutText =
(
	"Converts from a Daylight System with mental ray Sun and Sky to a Sun Positioner with a Physical sun and sky"
)

fn GetJulianDayNumberOfCurrentSoltice =
(
	--Starting julian day base on solstice of the current year
	local currentDate = getUniversalTime()
	local currentYear = currentDate[1]
	local month = 6 -- june
	local day = 21
	--This is based on formulas found on the julian day wikipedia https://en.wikipedia.org/wiki/Julian_day, except for the -0.5 at the end that come from the formula in max to take in account the conversion to integer
	local a = (integer) (floor ((14 - month) / 12)) 
	local y = (integer) (4800 + currentYear - a)
	local m = (integer) (month + (12 * a) - 3)  
	local julianDayOfSummerSolticeOfCurrentYear = (integer) (day + floor (((153 * m)+2)/5) + (365*y) + floor (y/4) - floor (y/100) + floor (y/400) - 32045 - 0.5) 

	julianDayOfSummerSolticeOfCurrentYear	
)


--This internal function does the parameter mapping
fn ConvertFromDaylightToSunPositioner legacyDaylight sunPos =
(
	if ((false == VerifySource legacyDaylight) or 
		(false == VerifyDestination sunPos)
		) do
	(
		--Not the suitable nodes
		return undefined
	)
	
	sunPos.name = legacyDaylight.name
	
	sunPos.sun_distance = legacyDaylight.controller.orbital_scale
	if(iskindof legacyDaylight.controller.orbital_scale.controller Bezier_Float) do sunPos.sun_distance.controller = legacyDaylight.controller.orbital_scale.controller
	
	--Sun position is determine by
	if(legacyDaylight.manual == 0) then -- date, time and location
	(

		--Date convert solar date which is a delta from the summer solstice of the current to julian days
		sunPos.julian_day = (GetJulianDayNumberOfCurrentSoltice() + legacyDaylight.controller.solar_date)
		--Time, we have to use solar_time.controller.value because solar_time give an integer instead of a float
		sunPos.time_in_seconds = legacyDaylight.controller.solar_time.controller.value * 60 * 60
		
		sunPos.dst = legacyDaylight.GetDaylightSavingTime()
		
		--Location
		sunPos.latitude_deg = legacyDaylight.controller.latitude
		if(iskindof legacyDaylight.controller.latitude.controller Bezier_Float) do
		(
			sunPos.latitude_deg.controller = legacyDaylight.controller.latitude.controller
		)
		sunPos.longitude_deg = legacyDaylight.controller.longitude
		if(iskindof legacyDaylight.controller.longitude.controller Bezier_Float) do
		(
			sunPos.longitude_deg.controller = legacyDaylight.controller.longitude.controller
		)
		
		sunPos.time_zone = legacyDaylight.GetTimeZone (legacyDaylight.controller.longitude)
		
	)
	else if(legacyDaylight.manual == 1) then --Manual
	(
		sunPos.mode = 0
		
		sunPos.manual_sun_position = legacyDaylight.controller.manual_position.position

		--Unused param on legacy daylight
		--legacyDaylight.controller.manual_position.rotation
		--legacyDaylight.controller.manual_position.scale
	)
	else if(legacyDaylight.manual == 2) do -- Weather data file
	(
		sunPos.mode = 2
		sunPos.weather_file = legacyDaylight.weatherFileName
	)
	
	sunPos.compass_radius = legacyDaylight.GetCompassDiameter()
	sunPos.north_direction_deg = legacyDaylight.GetNorthDirection()
	
	sunPos
)

fn ConvertMrPhysicalSkyToPhysicalSunSky mrMap physSunSky =
(
	if((false == (iskindof mrMap mr_Physical_Sky)) OR
	   (false == (iskindof physSunSky Physical_Sun___Sky_Environment))) do
	(
	   return undefined
	)
	
	physSunSky.sun_disc_intensity = mrMap.SunIntensity
	if(mrMap.SunIntensity.controller != undefined) do physSunSky.sun_disc_intensity.controller = mrMap.SunIntensity.controller
	
	physSunSky.sun_glow_intensity = mrMap.SunGlow
	if(mrMap.SunGlow.controller != undefined) do physSunSky.sun_glow_intensity.controller = mrMap.SunGlow.controller
	
	physSunSky.sun_disc_scale = mrMap.SunScale
	if(mrMap.SunScale.controller != undefined) do physSunSky.sun_disc_scale.controller = mrMap.SunScale.controller
	
	if (false == mrMap.InheritFromSky) do
	(
		physSunSky.global_intensity = mrMap.multiplier
		if(mrMap.multiplier.controller != undefined) do physSunSky.global_intensity.controller = mrMap.multiplier.controller

		physSunSky.sky_intensity = 1.0
					
		physSunSky.haze = mrMap.Haze / 15.0
		--if(mrMap.Haze.controller != undefined) do physSunSky.haze.controller = mrMap.Haze.controller
		
		physSunSky.night_color = mrMap.NightColor
		if(mrMap.NightColor.controller != undefined) do physSunSky.night_color.controller = mrMap.NightColor.controller
	
		physSunSky.night_intensity = 1.0	
		
		physSunSky.ground_color = mrMap.GroundColor
		if(mrMap.GroundColor.controller != undefined) do physSunSky.ground_color.controller = mrMap.GroundColor.controller
		
		physSunSky.horizon_height_deg = asin (mrMap.HorizonHeight * 0.1)
		physSunSky.horizon_blur_deg = asin (mrMap.HorizonBlur * 0.1)		
				
		physSunSky.saturation = mrMap.Saturation
		if(mrMap.Saturation.controller != undefined) do physSunSky.saturation.controller = mrMap.Saturation.controller 
		
		local c = Point4 (1.0 + mrMap.Redness) 1.0 (1.0 - mrMap.Redness) 1.0
		physSunSky.tint  = (c as Color)
	) 
)

fn InstallPhysicalSunAndSkyEnvMap sunPositioner =
(
--The code that install the environmap in the sun positioner, also configure the Exposure control, I don't know if this should also be done here.
--I am currently assuming that the exposure control will be set per another script.

	--Check if Physical Sun and Sky environment map is not already configured... 
	if(false == (iskindof environmentMap Physical_Sun___Sky_Environment)) do
	( -- if not, configure a new map

		local physSunSkyEnv = Physical_Sun___Sky_Environment name:Physical_Sun___Sky_Environment.localizedName --Not sure if it is the correct name	
		
		if(isKindOf environmentMap mr_Physical_Sky) then
		(
			ConvertMrPhysicalSkyToPhysicalSunSky environmentMap physSunSkyEnv
		)
		else 
		(
			--Other type of environmental map currently not supported for conversion, let take the default value
		)
		
		environmentMap = physSunSkyEnv 
	) 

	environmentMap.sun_position_object = sunPositioner
	
	environmentMap
)

fn ConvertLegacySunToPhysicalSun legacySun = 
(
	if(false == (isKindOf environmentMap Physical_Sun___Sky_Environment)) do 
	(
		return undefined
	)	 

	if(isKindOf legacySun mr_Sun) then
	(
		environmentMap.sun_enabled = legacySun.on
		--Param not converted shadows properties, photon settings, sun color (possibly inherited by the sky)
	)
	else
	(
		--We currently don't support conversion from other type of sun 
	)	
)

fn ConvertLegacySkyToPhysSky legacySky =
(
	if(false == (isKindOf environmentMap Physical_Sun___Sky_Environment)) do
	(
		return undefined
	)
	
	if(isKindOf legacySky mr_Sky) then
	(
		if(legacySky.on) do 
		(
			environmentMap.sky_intensity = 1.0

			environmentMap.ground_color = legacySky.GroundColor
			if(legacySky.GroundColor.controller != undefined) do environmentMap.ground_color.controller = legacySky.GroundColor.controller
			
			environmentMap.haze = 0
			environmentMap.perez_diffuse_horizontal_illuminance = 10000
			environmentMap.perez_direct_normal_illuminance = 10000
						
			if(legacySky.SkyLightModel == 0) then --Haze driven
			(
				environmentMap.illuminance_model = 1 --Physical
				
				environmentMap.haze = legacySky.Haze / 15.0
				if(legacySky.haze.controller != undefined) do environmentMap.haze.controller = legacySky.Haze.controller
			)
			else if(legacySky.SkyLightModel == 1) then --Perez All weather
			(
				environmentMap.illuminance_model = 2 --Perez All weather
				
				environmentMap.perez_diffuse_horizontal_illuminance = legacySky.PerezDiffuseHorizIlluminance
				if(legacySky.PerezDiffuseHorizIlluminance.controller) do environmentMap.perez_diffuse_horizontal_illuminance.controller = legacySky.PerezDiffuseHorizIlluminance.controller

				environmentMap.perez_direct_normal_illuminance = legacySky.PerezDirectNormalIlluminance
				if(legacySky.PerezDirectNormalIlluminance.controller) do environmentMap.perez_direct_normal_illuminance.controller = legacySky.PerezDirectNormalIlluminance.controller
			)
			else --CIE
			(
				environmentMap.illuminance_model = 0 --Automatic
				--for now this is not supported, setting to automatic			
			)
			
			environmentMap.horizon_height_deg = asin (legacySky.HorizonHeight * 0.1) 
			environmentMap.horizon_blur_deg = asin (legacySky.HorizonBlur * 0.1) 
			
			environmentMap.night_color = legacySky.NightColor
			if(legacySky.NightColor.controller != undefined) do environmentMap.night_color.controller = legacySky.NightColor.controller
			
			
			local c = Point4 (1.0 + legacySky.Redness) 1.0 (1.0 - legacySky.Redness) 1.0
			environmentMap.tint = (c as Color)
			
			environmentMap.saturation = legacySky.Saturation
			if(legacySky.Saturation.controller != undefined) do environmentMap.saturation.controller = legacySky.Saturation.controller
		)		
	)
	else
	(
		--We currently don't support other type of sky
	)	
)

--This function is use as entry when the source is missing (due to a missing plugin) and cannot be completly loaded. 
--In that case the default parameters are loaded.
fn DefaultConversion legacyDaylight =
(
	local sunPos = Sun_Positioner()
	
	newSun					= newNodesList[1]
	newSun.name	        	= sunPos.name --copy its name
	newSun.baseobject	 	= sunPos.baseobject
	
	--Delete the dummy physical light from the scene
	delete sunPos

	--Install a default physical sun and sky environment map
	InstallPhysicalSunAndSkyEnvMap newSun

	--return the new created light node
	newSun	
)

--Main entry point from this script
--This function handles the node's creation
fn Conversion legacyDaylight =
(
	if (false == VerifySource legacyDaylight)  do
	(
		--Not the suitable node
		return undefined
	)
	
	--We currently support only conversion from mental ray Sun and sky
	if((false == iskindof (legacyDaylight.sun) mr_Sun) OR (false == iskindof (legacyDaylight.sky) mr_Sky)) do
	(
		return undefined
	)
	
	--The legacy daylight should be compose of 4 objects in a 3 level hierachie, 
	-- A compass that acts as a base (at the top of the hierachie) 
	-- A daylight object (middle level)
	-- A Sun and a Sky (low level)	
	
	--We need the compass to determine the location  where the sun positioner will be created
	if(not isKindof legacyDaylight.parent Compass) do
    (		
		return undefined
	)	
	
	daylightCompass = legacyDaylight.parent
	
	--NOTE: I am not sure if cloning is a good idea here, from what I understand the point of cloning is the copy the controller,
	--but I am currently cloning only the compass to get the position correctly. A manual transfer of the 4 objects to the sun positioner might be better.
	
	--Clone the node to get all the node's data : wirecolor, transform and controllers etc.  
	--to be duplicated to the new node
	
	local ar = #() --the maxOps.CloneNodes needs an array of nodes
	ar[1] = daylightCompass
	
	--When you are cloning a light which has a target, the target will be cloned as well
	--That is what the actualNodesList contains, all cloned nodes from the node we provided.
	local actualNodesList = #()
	
	--The new cloned nodes
	local newNodesList 	 = #()
	
	--Clone our node
	local bRes = maxOps.CloneNodes ar actualNodeList:&actualNodesList newNodes:&newNodesList
	if (false == bRes)do return undefined --something went wrong	
			
	local sunPos = Sun_Positioner pos:(legacyDaylight.GetPosition())
	ConvertFromDaylightToSunPositioner legacyDaylight sunPos
	
	newSun					= newNodesList[1]
	newSun.name	        	= sunPos.name --copy its name
	newSun.baseobject	 	= sunPos.baseobject
	
	--Delete the dummy physical light from the scene
	delete sunPos
	
	--Transfer sun and sky from daylight to Physical Sun and Sky Map.
	InstallPhysicalSunAndSkyEnvMap newSun
	ConvertLegacySunToPhysicalSun  legacyDaylight.sun
	ConvertLegacySkyToPhysSky legacyDaylight.sky

	--return the new created light node
	newSun	
)